# Lenovo XClarity Proxmox_integrator
# Copyright 2025 Lenovo Corporation
# Author: Mario Sebastiani (msebastiani@lenovo.com)
#
# Version: 1.00 - 20250902 - First Release
#
# Licensed under the Apache License, Version 2.0 (the "License"); 
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software distributed under the License 
# is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
# either express or implied. See the License for the specific language governing permissions 
# and limitations under the License.

## Tested with 
# - LXCA 4.3.0
# - Proxmox 8.4.8

# Requirements: 
# Python3 (tested on Python 3.11.2, 3.13.5 (strict SSL))
# some external modules, see requirements.txt

import requests
from proxmoxer import ProxmoxAPI
import time
import configparser
import os
from ast import literal_eval
from collections import Counter
import urllib3
import logging
import getpass
from cryptography.fernet import Fernet, InvalidToken
import base64
import hashlib
import argparse
import sys
import paramiko

urllib3.disable_warnings()

################################################################################
## LOG
################################################################################
def log_initialize():
    formatter = logging.Formatter('[ %(asctime)s ], [ %(levelname)s ], [ %(process)d ][ %(message)s ], [ %(funcName)s():%(lineno)i ]')
    
    # if os.name == 'nt':
        # log_path = 'c:/temp/'
    # else:
        # home = expanduser('~')
        # log_path = home + '/logs/' 
        
    log_path = ""

    # Can be replaced with a config file parameter?
    log_name = "lxca_proxmox_integrator.log"
    
    mylogger_handler = logging.FileHandler(log_path + log_name, mode = 'a')    
    mylogger_handler.setFormatter(formatter)
    myloggerLog = logging.getLogger('mylogger')
    myloggerLog.setLevel(logging.DEBUG)
    myloggerLog.addHandler(mylogger_handler)

def log():
    return logging.getLogger('mylogger')

# def change(newLevel):
#     myloggerLog.setLevel(logging.newLevel)


################################################################################
## CONFIG
################################################################################

# Validate a configparser config file
def validate_config(config_path, required_structure):
    errors = []
    if not os.path.isfile(config_path):
        errors.append(f"File not found: {config_path}")
        return False, errors

    config = configparser.ConfigParser()
    try:
        config.read(config_path)
    except Exception as e:
        errors.append(f"Could not read config: {e}")
        return False, errors

    for section, keys in required_structure.items():
        if section not in config:
            errors.append(f"Missing section: {section}")
            continue
        for key in keys:
            if key not in config[section]:
                errors.append(f"Missing key in {section}: {key}")
    
    is_valid = len(errors) == 0
    return is_valid, errors

# Sec function to return key
def derive_key_pbkdf2(master_password: str, salt: bytes, iterations: int = 100_000) -> bytes:
    key = hashlib.pbkdf2_hmac('sha256', master_password.encode(), salt, iterations, dklen=32)
    return base64.urlsafe_b64encode(key)


################################################################################
## LXCA
################################################################################

# Retrieves managed servers UUIDs and returns an array
def get_lxca_servers(nodes): 
    lxca_servers = {}
    try:
        r = lxca_session.get(f"{BASE_URL}/nodes?status=managed", auth=lxca_credential, verify=LXCA_VERIFY_SSL)
        r.raise_for_status()
    except Exception as e:
        msg = f"Error while retrieving LXCA managed servers: {e}"
        print(msg)
        log().error(msg)
        exit(1)

    response_dict = r.json()
    for k in response_dict['nodeList']:
        for m in nodes:
            if k['mgmtProcIPaddress'] == m['xcc']:
                lxca_servers[k['mgmtProcIPaddress']] = k['uuid']
    
    return lxca_servers

# Compliance check (get latest compare result)
def node_needs_update(server_uuid):
    try:
        r = lxca_session.get(f"{BASE_URL}/compliancePolicies/persistedResult?type=SERVER&uuid={server_uuid}",auth=lxca_credential, verify=LXCA_VERIFY_SSL)
        r.raise_for_status()
    except Exception as e:
        msg = f"Error verifying if node {server_uuid} needs update: {e}"
        print(msg)
        log().error(msg)
        exit(1)
    
    resp = r.json()
    if resp['policyName'] == "":
        return False, ["No policy assigned to this host"]
    if resp['endpointCompliant'] == "no":
        # Create list of updates based on policy
        list_firmware = []

        for i in range(len(resp['targetFirmware'])):
            element = resp['targetFirmware'][i]
            if element['compliant'] == "no":
                list_firmware.append({"Fixid": element['fixid'], "Component": element['componentName']})
    
        return True, list_firmware
    else:
        return False, []

# Execute update on the server only with the compliant updates   
def trigger_firmware_update(server_uuid,list_updates):
    # Trigger update
    payload = {
        "DeviceList" : [{
            "ServerList" : [{
                "UUID" : server_uuid,
                "Components" : list_updates
            }]
        }]
    }
    try:
        r = lxca_session.put(f"{BASE_URL}/updatableComponents?action=apply&activationMode=immediate&forceUpdateMode=True", json=payload,auth=lxca_credential, verify=LXCA_VERIFY_SSL)
        r.raise_for_status()
    except Exception as e:
        msg = f"Error occurred in update task, see LXCA logs: {e}"
        print(msg)
        log().error(msg)
        exit(1)
   
   
    job_id = r.json()["JobID"]
    
    if VERBOSE:
        msg = f"Check on LXCA the running job {job_id}"
        print(f"\t{msg}")
        log().info(msg)

    # The JOB API seems broken, it returns always 500 Internal Server Error
    #
    # Wait for completion (poll)
    timeout_host = TIMEOUT_HOST
    while timeout_host > 0:
        try:
            r = lxca_session.get(f"{BASE_URL}/tasks/{job_id}", auth=lxca_credential, verify=LXCA_VERIFY_SSL)
            r.raise_for_status()
            resp = r.json()
            status = resp[0]['status']
            if status in ["Complete"]:
                msg = f"Upgrade procedure is {status} on this node"
                print(f"\t{msg}")
                log().info(msg)
                timeout_host = 0
                break
            elif status in ["CompleteWithWarning", "CompleteWithError"]:
                msg = f"Upgrade procedure is {status}, please check logs"
                print(f"\t{msg}")
                log().warning(msg)
                timeout_host = 0
                break
            elif status in ["Pending","Running","RunningWithError", "RunningWithWarning"]:
                msg = f"JobID {job_id} still running... wait to complete"
                print(f"\t{msg}")
                if VERBOSE:
                    log().debug(msg)
                time.sleep(10)
                timeout_host -= 10
            else:
                msg = "Something went wrong with the update, please check logs"
                print(msg)
                log().error(msg)
                exit(1)
                
        except Exception as e:
            msg = f"Error occurred getting job status - Stopping procedure: {e}"
            print(msg)
            log().error(msg)
            exit(1)



################################################################################
## PROXMOX
################################################################################

# Connection to the Proxmox API backend
def prox_session(node):
    try:
        return ProxmoxAPI(node, user=PROXMOX_USER, password=PROXMOX_PASS, verify_ssl=PROXMOX_VERIFY_SSL, timeout=10)
    except Exception as e:
        msg = f"Error connecting to Proxmox node: {e}"
        print(msg)
        log().error(msg)
        exit(1)


# Verify node status
def node_status(session, node,timeout=60):
    status = False
    while timeout > 0:
        try:
            cluster_status = session.cluster.status.get()
            break    
        except:
            time.sleep(10)
            timeout -= 10
    for entry in cluster_status:
        if entry.get('type') == 'node' and entry.get('name') == node:
                status = entry.get('online', 0) == 1
                break
    return status

# Create a list with the running VMs/containers on a specific node
def get_running(session,node,type):
    mylist = []
    try:
        if type == "vm":
            for vm in session.nodes(node).qemu.get():
                if vm['status'] == "running":
                    mylist.append(vm['vmid'])
        else:
            for ct in session.nodes(node).lxc.get():
                if ct['status'] == "running":
                    mylist.append(ct['vmid'])
    except Exception as e:
        if type == "vm":
            msg = f"Error while retrieving VM running on {node}: {e}"
        else:
            msg = f"Error while retrieving containers running on {node}: {e}"
        print(f"\t{msg}")
        log().error(msg)
        exit(1)

    return mylist


# Poweroff VM/container
def shutdown_single(session,node,vmid, type):
    if not DRY_RUN:
        try:
            if type == "vm":
                session.nodes(node).qemu(vmid).status.shutdown.post(forceStop=1,skiplock=1,timeout=TIMEOUT_VM)
            else:
                session.nodes(node).lxc(vmid).status.shutdown.post(forceStop=1,timeout=TIMEOUT_VM)
        except Exception as e:
            if type == "vm":
                msg = f"Error occurred shutting down VMs- Stopping procedure. Error: {e}"
            else:
                msg = f"Error occurred shutting down containers- Stopping procedure. Error: {e}"
            print(msg)
            log().error(msg)
            exit(1)

# Shutdown all the list of running VMS/Containers
def shutdown(session, node, vm_list, type):
    for vmid in vm_list:
        if VERBOSE:
            msg = f"Shutting down {vmid}"
            print(f"\t{msg}")
            log().debug(msg)
        if not DRY_RUN:
            shutdown_single(session, node, vmid,type)
        

# Check poweroff on node for a list of VMs/containers
def check_shutdown(session, node, vm_list, type, list_type):
    vm_timeout = TIMEOUT_VM
    while vm_timeout > 0:
        if type == "vm":
            running_list = get_running(session, node, "vm")
        else:
            running_list = get_running(session, node, "ct")

        vm_found = False
        if list_type == "object":
            for i in range(len(vm_list)):
                if vm_list[i]['vmid'] in running_list:
                    vm_found = True
                    time.sleep(5)
                    vm_timeout -= 5
                    if vm_timeout <= 0:
                        return False
                    continue
            if vm_found == False:
                return True
        elif list_type == "vmid":
            for vm in vm_list:
                if vm in running_list:
                    vm_found = True
                    time.sleep(5)
                    vm_timeout -= 5
                    if vm_timeout <= 0:
                        return False
                    continue
            if vm_found == False:
                return True
    return False


# Power on single VM/container
def poweron_single(session,node,vmid,type):
    if not DRY_RUN:
        try:
            if type == "vm":
                session.nodes(node).qemu(vmid).status.start.post()
            else:
                session.nodes(node).lxc(vmid).status.start.post()
        except Exception as e:
            if type == "vm":
                msg = f"Error powering on VM {vmid} on node {node}: {e}"
            else:
                msg = f"Error powering on container {vmid} on node {node}: {e}"
            print(f"\t{msg}")
            log().error(msg)

# Start VM/container in a list
def poweron(session, node, vm_list, type):
    for vmid in vm_list:
        if VERBOSE:
            msg = f"Power on {vmid} on node {node}"
            print(f"\t{msg}")
            log().debug(msg)
        poweron_single(session,node,vmid,type)


## Check poweron on node for a list of VMs/containers
def check_poweron(node, vm_list, type):
    vm_timeout = TIMEOUT_VM
    while vm_timeout > 0:
        session = prox_session(node_ip)
        if type == "vm":
            running_list = get_running(session, node, "vm")
        else:
            running_list = get_running(session, node, "ct")
        if all(e in running_list for e in vm_list):
            return True
        time.sleep(5)
        vm_timeout -= 5
        if vm_timeout <= 0:
            return False
    

            
# Migrate VMs/Containers
def migrate(session, type,placement_list, host, direction):

    for i in range(len(placement_list)):
        vmid = placement_list[i][type]
        if direction == "Forward":
            node = host
            tgt_node = placement_list[i]['host']
        else:
            node = placement_list[i]['host']
            tgt_node = host
        if VERBOSE:
            if type == "vm":
                msg = f"Migrating VM {vmid} from {node} to {tgt_node}"
            else:
                msg = f"Migrating container {vmid} from {node} to {tgt_node}"
            print(f"\t{msg}")
            log().debug(msg)

        if not DRY_RUN:
            try:
                if type == "vm":
                    my_parameters = {'target': tgt_node, 'online': 1, 'with-local-disks': 1}
                    session.nodes(node).qemu(vmid).migrate.post(**my_parameters) 
                else:
                    session.nodes(node).lxc(vmid).migrate.post(target=tgt_node, restart=1)
            except Exception as e:
                msg = f"An error occurred during the migration: {e}"
                print(f"\t{msg}")
                log().error(msg)
                exit(1)

# Create the sequential placement list for VMs migration
def create_placement_list(session, vm_list,host_list, type):
    k = 0
    j = 0
    placement_list = []
    while k < len(vm_list):
        if j >= len(host_list):
            j=0       
        if type == "vm":
            placement_list.append({"host" : host_list[j],"vm" : vm_list[k]})
        else:
            placement_list.append({"host" : host_list[j],"ct" : vm_list[k]})
        j += 1
        k += 1
    return placement_list



# Change maintenance mode on the host
def change_maintenace(host_name, host_ip, status):
    command = f"ha-manager crm-command node-maintenance {status} {host_name}"

    client = paramiko.client.SSHClient()
    client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
    client.connect(host_ip, username=PROXMOX_USER.replace("@pam",""), password=PROXMOX_PASS)

    _stdin, _stdout,_stderr = client.exec_command(command)
    err = _stderr.readlines()
    if len(err) > 0:
        msg = f"An error occurred while changing maintenance mode on {host_name}: {err}"
        print(msg)
        log().error(msg)
        exit(1)
    client.close()

# Determine if nodes are in a cluster. Return is_cluster (boolean) and node list dict (node name, node ip)
def is_cluster(nodeslist):
    mynode = {}
    cluster_nodes = []
    is_cluster = False
    for i in nodeslist:
        if i['type'] == "cluster":
            is_cluster = True
            continue
        if i['type'] == "node":
            mynode['name'] = i['name']
            mynode['ip'] = i['ip']
            cluster_nodes.append(mynode)
            mynode = {}
    return is_cluster, cluster_nodes


def get_cluster_status(node_ip):
    status = []
    try:
        # proxmox = ProxmoxAPI(node_ip, user=PROXMOX_USER, password=PROXMOX_PASS, verify_ssl=PROXMOX_VERIFY_SSL, timeout=10)
        proxmox = prox_session(node_ip)
        status = proxmox.cluster.status.get()
    except Exception as e:
        msg = f"An error occurred trying to retrieve cluster status: {e}"
        print(msg)
        log().error(msg)
        exit(1)
    return status


def check_cluster_ok(nodeslist, cluster_nodes):
    passed_nodes = []
    mynode = {}
    for i in range(len(nodeslist)):
        mynode['name'] = nodeslist[i]['pve_host']
        mynode['ip'] = nodeslist[i]['pve_ip']
        passed_nodes.append(mynode)
        mynode = {}
    
    mynode = {}
    mycluster_nodes = []
    for i in cluster_nodes:
        if i['type'] == "cluster":
            is_cluster = True
            continue
        if i['type'] == "node":
            mynode['name'] = i['name']
            mynode['ip'] = i['ip']
            mycluster_nodes.append(mynode)
            mynode = {}

    set_passed_nodes = {frozenset(d.items()) for d in passed_nodes}
    set_mycluster_nodes = {frozenset(d.items()) for d in mycluster_nodes}
    
    if set_passed_nodes == set_mycluster_nodes:
        return True
    else:
        if VERBOSE:
            msg = f"Proxmox config from node: {cluster_nodes}"
            print(f"\n{msg}")
            log().debug(msg)
        return False

# Get the list of the VMs managed by HA
def get_ha_managed_vms(node):
    vm_list = []
    try:
        proxmox = ProxmoxAPI(node, user=PROXMOX_USER, password=PROXMOX_PASS, verify_ssl=PROXMOX_VERIFY_SSL, timeout=10)
        ha_status = proxmox.cluster.ha.status.current.get()
        for i in range(len(ha_status)):
            myvm = {}
            if ha_status[i]['type'] == 'service':
                myvm['node'] = ha_status[i]['node']
                myvm['sid'] = ha_status[i]['sid']
                myvm['state'] = ha_status[i]['state']
                vm_list.append(myvm)
    except Exception as e:
        msg = f"An error occurred trying to retrieve cluster status: {e}"
        print(msg)
        log().error(msg)
        exit(1)
    return vm_list

# Check the completion of the migration 
# Direction:  "off" move away from this host, "on" move on this host
def check_migration_ha(session, node_ip,node_name,ha_vm_list,direction):
    vm_timeout = TIMEOUT_VM
    vm_list = []
    ha_vm_found = []
    count_ha_vm = len(ha_vm_list)
    while vm_timeout > 0:
        try:
            session = prox_session(node_ip)
            cluster_status = session.cluster.ha.status.current.get()
        except Exception as e:
            msg = f"Error during querying HA status: {e}"
            print(f"\t{msg}")
            log().error(msg)
            exit(1)

        if direction == "off":
            counter = 0
            for i in range(len(cluster_status)):
                if cluster_status[i]["node"] == node_name and cluster_status[i]['type'] == "service" and cluster_status[i]['state'] == "started":
                    counter += 1
                if cluster_status[i]['type'] == "service" and (cluster_status[i]['state'] == "migrate" or cluster_status[i]['state'] == "relocate"):
                    counter +=1
            if counter == 0:
                vm_timeout = 0
                continue
            time.sleep(5)
            vm_timeout -= 5
            if vm_timeout <= 0:
                msg = f"HA-Migration failed. Some HA protected VM/container still running on {node_name} after waiting a timeout of {TIMEOUT_VM} seconds"
                print(f"\t{msg}")
                log().error(msg)
                exit(1)
        
        if direction == "on":
            counter = 0
            for i in range(len(cluster_status)):
                if cluster_status[i]["node"] == node_name and cluster_status[i]['type'] == "service" and cluster_status[i]['state'] == "started":
                    counter += 1
            if counter == count_ha_vm:
                vm_timeout = 0
                continue
            time.sleep(5)
            vm_timeout -= 5
            if vm_timeout <= 0:
                msg = f"HA-Migration failed. Not all the HA protected VM/container were restored on {node_name} after waiting a timeout of {TIMEOUT_VM} seconds"
                print(f"\t{msg}")
                log().error(msg)
                exit(1)
                
# Verify if VM has local resources and/or local disk
# Returns boolean localdisks, localresources, localcdrom and a list with all the VMs attribute
def check_local(node_ip, node_name, vm_info):
    localdisks = False
    localresources = False
    localcdrom = False
    try:
        proxmox = prox_session(node_ip)
        vm_attr = proxmox.nodes(node_name).qemu(vm_info['vmid']).migrate.get()
        if len(vm_attr['local_disks']) > 0:
            for i in vm_attr['local_disks']:
                if i['cdrom'] == 1:
                    localcdrom = True
                else:
                    localdisks = True
        if len(vm_attr['local_resources']) > 0:
            localresources = True
        return localdisks, localresources, localcdrom, vm_attr
    except Exception as e:
        msg = f"Error querying VM {vm_attr['name']} local resources on node {node_name}: e"
        print(f"\t{msg}")
        log().error(msg)
        

###########################################################
## MAIN
###########################################################
if __name__ == "__main__":
    VERSION = "1.00"
    __author__ = "Mario Sebastiani <msebastiani@lenovo.com>"

    parser = argparse.ArgumentParser(description="LXCA Proxmox integrator command line parameters")

    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument("-d", "--dry-run", help="Simulate the upgrade",action='store_true')
    group.add_argument("-x", "--execute",  help="Perform the upgrade", action='store_true')
    parser.add_argument("-v","--verbose", help="Add verbosity to the ouput", action='store_true')
    parser.add_argument("-c","--clearlog", help="Move current log to backup and start a clean log file", action="store_true")
    parser.add_argument("-V", "--version", help="Show program version", action="version", version=f'{VERSION}')

    if len(sys.argv) == 1:
        parser.print_help()
        exit(0)

    
    args = parser.parse_args() 

    if args.clearlog:
        if os.path.isfile("lxca_proxmox_integrator.log.bck"):
            try:
                os.remove("lxca_proxmox_integrator.log.bck")
            except Exception as e:
                print(f"Error deleting log backup file: {e}")
                exit(1)
        if os.path.isfile("lxca_proxmox_integrator.log"):
            try:
                os.rename("lxca_proxmox_integrator.log", "lxca_proxmox_integrator.log.bck")
                # Log Initialize
                log_initialize()
                msg ="Moving log file to backup"
                print(msg)
                log().info(msg)
            except Exception as e:
                msg = f"Error moving log file to backup: {e}"
                # Log Initialize
                log_initialize()
        else:
            # Log Initialize
            log_initialize()
    else:
        # Log Initialize
        log_initialize()
    
        
    if args.execute:
        DRY_RUN = False
    else:
        DRY_RUN = True

    if not DRY_RUN:
        while True:
            print("\nDid you already execute a dry-run round to verify what will happen?")
            response = input("\nPlease, confirm that you want to proceed with the update (Y/[N]) ")
            if response.lower() == "y":
                break
            elif response.lower() == "n" or not response:
                exit(0)
            else:
                continue
        while True:
            response = input("\n\n**ATTENTION** - Do you want to stop now this procedure ([Y]/N) ")
            if response.lower() == "y" or not response:
                exit(0)
            elif response.lower() == "n":
                break
            else:
                continue
    
    msg = "Starting program"   
    print(msg)
    log().info(msg)

    VERBOSE = args.verbose
    msg = f"Command line parameters: DRY_RUN = {DRY_RUN}, VERBOSE = {VERBOSE}"
    print(msg)
    log().info(msg)

    # Specify the config file name
    config_file = "config.ini"

    msg = "Validating configuration file"
    print(msg)
    log().info(msg)
    
    # Verify config file
    required = {
        'Proxmox_integrator': ['PROXMOX_USER', 'PROXMOX_PASS','PROXMOX_VERIFY_SSL',
                               'CLUSTER_NODES','LXCA_HOST','LXCA_USER','LXCA_PASS','LXCA_VERIFY_SSL']
    }

    is_valid, error_list = validate_config(config_file, required)
    if not is_valid:
        msg = "Config validation errors:"
        print(msg)
        log().error(msg)
        for e in error_list:
            print(f"- {e}")
            log().error(e)
            exit(1)
    else:
        msg = "Config file is valid!"
        print(msg)
        log().info(msg)

    # Read the parameters
    config = configparser.ConfigParser()
    config.read(config_file)
   
    myconfig = config['Proxmox_integrator']
    
    msg = "Loading configuration"
    print(msg)
    log().info(msg)

    # Check if custom certificate are used
    if os.path.isfile('custom_cacert.pem'):
        os.environ['REQUESTS_CA_BUNDLE'] = 'custom_cacert.pem'
    else:
        os.environ['REQUESTS_CA_BUNDLE'] = ''

    # Determine if encryption was used for stored password
    if config.has_option('Proxmox_integrator','SALT'):
        ENCRYPTION = True
        master_password = getpass.getpass("Enter master password to decrypt stored password: ")
        salt = base64.b64decode(myconfig['SALT'])
        key = derive_key_pbkdf2(master_password, salt)
        fernet = Fernet(key)
    else:
        ENCRYPTION = False

    # Get the TIMEOUT value if present, otherwise default is 3600 secs (1H) for HOST operations, 300 secs (5M) for VM operations
    if config.has_option('Proxmox_integrator','TIMEOUT_HOST'):
        TIMEOUT_HOST = myconfig.getint('TIMEOUT_HOST')
    else:
        TIMEOUT_HOST = 3600 

    if config.has_option('Proxmox_integrator','TIMEOUT_VM'):
        TIMEOUT_VM = myconfig.getint('TIMEOUT_VM')
    else:
        TIMEOUT_VM = 300
    
    

    # Proxmox setup
    PROXMOX_USER = myconfig['PROXMOX_USER']
    if ENCRYPTION:
        try:
            PROXMOX_PASS = fernet.decrypt(myconfig['PROXMOX_PASS'].encode()).decode()
        except InvalidToken:
            msg = "Invalid master password or corrupted encrypted PROXMOX_PASS"
            log().error(msg)
            raise ValueError(msg)
    else:
        PROXMOX_PASS = myconfig['PROXMOX_PASS']
    
    try:
        PROXMOX_VERIFY_SSL = myconfig.getboolean('PROXMOX_VERIFY_SSL')
    except:
        PROXMOX_VERIFY_SSL = False

    if VERBOSE:
        print("\nProxmox access parameters from config file:")
        print("===========================================")
        print(f"Proxmox user: {PROXMOX_USER}")
        print(f"Proxmox password: {PROXMOX_PASS}")
        print(f"Verify SSL cert: {PROXMOX_VERIFY_SSL}")
        log().debug(f"Proxmox access parameters from config file - user: {PROXMOX_USER}, verify SSL cert: {PROXMOX_VERIFY_SSL}")

    # Declare what CA BUNDLE is used
    if VERBOSE:
        if os.environ['REQUESTS_CA_BUNDLE'] != "":
            msg = f"Using this CA file: {os.environ['REQUESTS_CA_BUNDLE']}"
        else:
            msg = f"Using this CA file: {requests.utils.DEFAULT_CA_BUNDLE_PATH}"

        print(f"\n{msg}")
        log().info(msg)


    # Get the LOCAL_DISK and LOCAL_RESOURCES behavior
    # Default = FAIL
    # Option: FAIL (stop the procedure) - POWEROFF (shutdown the VM with local resources/disks)
    if config.has_option('Proxmox_integrator','VM_LOCAL_RESOURCES'):
        VM_LOCAL_RESOURCES = myconfig['VM_LOCAL_RESOURCES'].upper()
        if VM_LOCAL_RESOURCES != "POWEROFF" and VM_LOCAL_RESOURCES != "FAIL":
            msg = f"Error validating VM_LOCAL_RESOURCE field: {VM_LOCAL_RESOURCES}"
            print(msg)
            log().error(msg)
            exit(1)
        msg = f"VM_LOCAL_RESOURCES from config file = {VM_LOCAL_RESOURCES}"
    else:
        VM_LOCAL_RESOURCES = "FAIL"
        msg = f"VM_LOCAL_RESOURCES missing from config file, set it to default = {VM_LOCAL_RESOURCES}"

    if VERBOSE:
        print(f"\n{msg}")
        log().debug(msg)

    if config.has_option('Proxmox_integrator',"VM_LOCAL_DISKS"):
        VM_LOCAL_DISKS = myconfig['VM_LOCAL_DISKS'].upper()
        if VM_LOCAL_DISKS != "POWEROFF" and VM_LOCAL_DISKS != "FAIL":
            msg = f"Error validating VM_LOCAL_DISKS field: {VM_LOCAL_DISKS}"
            print(msg)
            log().error(msg)
            exit(1)
        msg = f"VM_LOCAL_DISKS from config file = {VM_LOCAL_DISKS}"
    else:
        VM_LOCAL_DISKS = "FAIL"
        msg = f"VM_LOCAL_DISKS missing from config file, set it to default = {VM_LOCAL_DISKS}"

    if VERBOSE:
        print(f"{msg}")
        log().debug(msg)

    if config.has_option('Proxmox_integrator',"VM_LOCAL_DISKS_EXPERT"):
        try:
            VM_LOCAL_DISKS_EXPERT = myconfig.getboolean('VM_LOCAL_DISKS_EXPERT')
        except:
            VM_LOCAL_DISKS_EXPERT = False
        msg = f"VM_LOCAL_DISKS_EXPERT from config file = {VM_LOCAL_DISKS_EXPERT}"
    else:
        VM_LOCAL_DISKS_EXPERT = False
        msg = f"VM_LOCAL_DISKS_EXPERT missing from config file, set it to default = {VM_LOCAL_DISKS_EXPERT}"

    if VERBOSE:
        print(f"{msg}")
        log().debug(msg)
    
    # cluster node list
    nodes = literal_eval(myconfig['CLUSTER_NODES'])
    if len(nodes) < 1:
        msg = "Error in the config file. CLUSTER_NODES field is empty (or incorrectly filled in)"
        print(msg)
        log().error(msg)
        exit(1)
    elif len(nodes) == 1:
        SINGLE_NODE = True
    else:
        SINGLE_NODE = False

    if VERBOSE:
        if SINGLE_NODE:
            msg = "Proxmox single node from config file:"
        else:
            msg = "Proxmox cluster parameters from config file:"
        print(f"\n{msg}")
        log().debug(msg)
        print("============================================")
        print("{:<20} {:<20} {:<20}".format("PVE Host", "PVE IP", "XCC host"))
        for i in range(len(nodes)):
            print("{:<20} {:<20} {:<20}".format(nodes[i]['pve_host'], nodes[i]['pve_ip'], nodes[i]['xcc']))
            log().debug(f"PVE node: {nodes[i]['pve_host']}, PVE IP: {nodes[i]['pve_ip']}, XCC IP: {nodes[i]['xcc']}")

    # LXCA setup
    LXCA_HOST = myconfig['LXCA_HOST']
    if ENCRYPTION:
        try:
            LXCA_PASS = fernet.decrypt(myconfig['LXCA_PASS'].encode()).decode()
        except InvalidToken:
            raise ValueError("Invalid master password or corrupted encrypted LXCA_PASS")
    else:
        LXCA_PASS = myconfig['LXCA_PASS']

    LXCA_USER = myconfig['LXCA_USER']
    try:
        LXCA_VERIFY_SSL = myconfig.getboolean('LXCA_VERIFY_SSL')
    except:
        LXCA_VERIFY_SSL = False

    BASE_URL    = f"https://{LXCA_HOST}"  # Adjust to use correct port if necessary

    if VERBOSE:
        msg = "Lenovo XClarity Administrator parameters"
        print(f"\n{msg}")
        print("========================================")
        log().debug(msg)
        log().debug(f"LXCA Host: {LXCA_HOST}, LXCA User: {LXCA_USER}, Verify SSL cert: {LXCA_VERIFY_SSL}")
        log().info(f"We will use this URL: {BASE_URL}")
        print(f"LXCA Host: {LXCA_HOST}")
        print(f"LXCA User: {LXCA_USER}")
        print(f"LXCA Password: {LXCA_PASS}")
        print(f"Verify SSL cert: {LXCA_VERIFY_SSL}")
        print(f"We will use this URL: {BASE_URL}\n")

    # DRY run or execute
    if VERBOSE:
        if DRY_RUN:
            msg = "DRY_RUN = True ---------> No action will be really performed, only simulation"
        else:
            msg = "DRY_RUN = False --------> Executing update (if needed)"
            
        print(f"\n{msg}")
        log().info(msg)

    # Setting up 
    lxca_credential = (LXCA_USER, LXCA_PASS)

    lxca_session = requests.Session()
    lxca_session.verify = LXCA_VERIFY_SSL  # Consider security
    lxca_session.headers.update({"Content-Type": "application/json"})
    if VERBOSE:
        msg = f"HEADERS for all LXCA requests: {lxca_session.headers}"
        print(f"\n{msg}")
        log().debug(f"{msg}")

    msg = "Creating list with host UUID from LXCA"
    print(f"\n{msg}")
    log().info(msg)

    # Create list with host UUID
    my_lxca_servers = get_lxca_servers(nodes)
    if VERBOSE:
        if len(my_lxca_servers) > 1:
            msg = f"These are the servers managed by LXCA: {my_lxca_servers}"
        else:
            msg = f"This is the server managed by LXCA: {my_lxca_servers}"
        print(msg)
        log().debug(msg)

    # Verify Proxmox cluster
    cluster_status = []
    if len(nodes) == 1:
        cluster_ok = is_cluster = False
    else:
        cluster_status = get_cluster_status(nodes[0]['pve_ip'])
        cluster_ok = check_cluster_ok(nodes, cluster_status)
        is_cluster, cluster_nodes = is_cluster(cluster_status)
    
    if not is_cluster:
        msg = "No cluster found"
        print(msg)
        log().info(msg)

    if is_cluster and not cluster_ok:
        msg = "Error in cluster definition. Please check consistency between config file and your Proxmox cluster"
        print(f"\n{msg}")
        log().error(msg)
        exit(1)

    # Grab info about HA managed VMs
    my_ha_vm_list = []
    if is_cluster:
        my_ha_vm_list = get_ha_managed_vms(nodes[0]['pve_ip'])
    
    
    # Create list of Proxmox nodes
    my_pve_hosts = []
    for i in range(len(nodes)):
        my_pve_hosts.append(nodes[i]["pve_host"])

    # Verify if VMs have local disks and/or local resources
    msg = "Checking for local resources/local disks"
    print(f"\n{msg}")
    log().info(msg)

    vm_local_disks = []
    vm_local_disks_vmid = []
    vm_local_resources = []
    vm_local_resources_vmid = []
    vm_local_cdrom = []
    vm_local_cdrom_vmid = []
    vm_attributes_list = []
    for i in range(len(nodes)):
        try:
            proxmox = prox_session(nodes[i]['pve_ip'])
            node = nodes[i]['pve_host']
            vm_node_list = proxmox.nodes(node).qemu.get()
        except Exception as e:
            msg = f"Error getting VM list on {node}"
            print(msg)
            log().error(msg)
            exit(1)
        for k in range(len(vm_node_list)):
            my_vm_info = {}
            local_disks = local_resources = local_cdrom = False
            if vm_node_list[k]['status'] == "running":
                local_disks, local_resources, local_cdrom, vm_attributes = check_local(nodes[i]['pve_ip'],node,vm_node_list[k])
                # Add vm to the global vm_attributes_list
                vm_attributes_list.append(vm_attributes)

                if local_resources:
                    my_vm_info['name'] = vm_node_list[k]['name']
                    my_vm_info['vmid'] = vm_node_list[k]['vmid']
                    my_vm_info['node'] = node
                    vm_local_resources.append(my_vm_info)
                    vm_local_resources_vmid.append(my_vm_info['vmid'])
                    my_vm_info = {}
                
                if local_disks:
                    for ld in vm_attributes['local_disks']:
                        if ld['cdrom'] == 1:
                            my_vm_info['name'] = vm_node_list[k]['name']
                            my_vm_info['vmid'] = vm_node_list[k]['vmid']
                            my_vm_info['node'] = node
                            vm_local_cdrom.append(my_vm_info)
                            vm_local_cdrom_vmid.append(my_vm_info['vmid'])
                            my_vm_info = {}
                        else:
                            my_vm_info['name'] = vm_node_list[k]['name']
                            my_vm_info['vmid'] = vm_node_list[k]['vmid']
                            my_vm_info['node'] = node
                            vm_local_disks.append(my_vm_info)
                            vm_local_disks_vmid.append(my_vm_info['vmid'])
                            my_vm_info = {}
                   
    vm_local_cdrom_vmid.sort()
    vm_local_disks_vmid.sort()
    vm_local_resources_vmid.sort()
    # Verify VM_LOCAL_DISK policy against VM with cdrom
    if VM_LOCAL_DISKS != "POWEROFF" and len(vm_local_cdrom) > 0:
        msg = f"Found VM with local cdrom: {vm_local_cdrom_vmid} that prevent live migration. Requested behavior is to stop procedure"
    elif len(vm_local_cdrom) > 0:
        msg = f"Found VM with local cdrom: {vm_local_cdrom_vmid} that prevent live migration. Requested behavior is to poweroff VM"
    else:
        msg = "No VM with local cdrom"
    
    print(f"\n{msg}")
    if VM_LOCAL_DISKS != "POWEROFF" and len(vm_local_cdrom) > 0:
        log().error(msg)
        exit(1)
    else:
        log().info(msg)

    # Verify VM_LOCAL_DISKS policy against VM with local disks
    if VM_LOCAL_DISKS != "POWEROFF" and len(vm_local_disks) > 0:
        msg = f"Found VM with local disks: {vm_local_disks_vmid} that prevent live migration. Requested behavior is to stop procedure"
    elif len(vm_local_disks) > 0:
        msg = f"Found VM with local disks: {vm_local_disks_vmid} that prevent live migration. Requested behavior is to poweroff VM"
    else:
        msg = "No VM with local disks"
        
    print(f"\n{msg}")
    if VM_LOCAL_DISKS != "POWEROFF" and len(vm_local_disks) > 0:
        log().error(msg)
        exit(1)
    else:
        log().info(msg)
    
    if VM_LOCAL_RESOURCES != "POWEROFF" and len(vm_local_resources) > 0:
        msg = f"Found VM with local resources: {vm_local_resources_vmid} that prevent live migration. Requested behavior is to stop procedure"
    elif len(vm_local_resources) > 0:
        msg = f"Found VM with local resources: {vm_local_resources_vmid} that prevent live migration. Requested behavior is to poweroff VM"
    else:
        msg = "No VM with local resources"

    print(f"\n{msg}")
    if VM_LOCAL_RESOURCES != "POWEROFF" and len(vm_local_resources) > 0:
        log().error(msg)
        exit(1)
    else:
        log().info(msg)

    
    # Execute the loop
    msg = "Starting the loop"
    print(f"\n{msg}")
    log().info(msg)

    for i in range(len(nodes)):
        
        node = nodes[i]["xcc"]
        node_pve = nodes[i]["pve_host"]
        node_ip = nodes[i]["pve_ip"]

        if node in my_lxca_servers.keys():
            uuid = my_lxca_servers[node]
        else:
            uuid = None
            msg = f"Node {node} not found in LXCA, skipping."
            print(msg)
            log().warning(msg)
            continue

        if VERBOSE:
            msg = f"Checking {node_pve} that has IP: {node_ip} and XCC IP: {node} with UUID: {uuid}"
            log().debug(msg)
        else:
            msg = f"Checking {node_pve}"
            log().info(msg)
        
        print(msg)
        
        msg = "Verifying if node needs update"
        print(f"\t{msg}")
        log().info(msg)

        # Verify if node needs update and return, if needed, the list of updates 
        needs_update = False
        list_updates = []
        needs_update, list_updates = node_needs_update(uuid)
        if not needs_update:
            if list_updates == ["No policy assigned to this host"]:
                msg = f"The node {node_pve} has no associated policy in LXCA. Skipping..."
            elif list_updates == []:
                msg = f"{node_pve} is up-to-date. Skipping..."
            print(f"\t{msg}")
            log().info(msg)
            continue
        
        if not DRY_RUN:
            msg = f"Starting upgrade procedure on {node_pve}"
        else:
            msg = f"Simulating upgrade procedure on {node_pve}"
        
        print(f"\t{msg}")
        log().info(msg)

        # Evacuate VMs
        try:
            proxmox = prox_session(node_ip)
        except Exception as e:
            print(e)
            log().error(e)
            exit(1)

        msg = f"Getting the list of running VMs on {node_pve}"
        print(f"\t{msg}")
        log().info(msg)
        
        # Create list of running VMs
        vm_running_list = get_running(proxmox, node_pve, "vm")
        vm_running_list.sort()

        if VERBOSE:
            if len(vm_running_list) > 1:
                msg = f"These are the running VMs: {vm_running_list}"
            elif len(vm_running_list) == 1:
                msg = f"This is the running VM: {vm_running_list.sort()}"
            else:
                msg = "No running VMs found"
                
            print(f"\t{msg}")
            log().debug(msg)  

        # Get the running VM list managed by HA on this node
        thisnode_ha_vms = []
        if is_cluster:
            for i in range(len(my_ha_vm_list)):
                if my_ha_vm_list[i]['node'] == node_pve and my_ha_vm_list[i]['state'] == 'started':
                    thisnode_ha_vms.append(my_ha_vm_list[i]['sid'])
        thisnode_ha_vms.sort()

        # List VMs with local cdrom
        this_node_local_cdrom = []
        for vm in vm_local_cdrom:
            if vm['node'] == node_pve:
                this_node_local_cdrom.append(vm['vmid'])
        this_node_local_cdrom.sort()
        
        # Shutdown VM with local cdrom if present
        if len(this_node_local_cdrom) > 0:
            if VERBOSE:
                msg = f"Powering off VM with local cdrom: {this_node_local_cdrom}"
            else:
                msg = f"Powering off VM with local cdrom"

            print(f"\t{msg}")
            log().debug(msg)
            for i in range(len(vm_local_cdrom)):
                if vm_local_cdrom[i]['node'] == node_pve:
                    msg = f"Powering off VM {vm_local_cdrom[i]['vmid']} - {vm_local_cdrom[i]['name']}"

                    if VERBOSE:
                        print(f"\t{msg}")
                        log().debug(msg)

                    shutdown_single(proxmox, node_pve,vm_local_cdrom[i]['vmid'], "vm")

                    # Remove the VM from the vm_local_disks if present because already powered off
                    if vm_local_cdrom[i] in vm_local_disks:
                        vm_local_disks.remove(vm_local_cdrom[i])
                    
                    # Remove the VM from the vm_local_resources if present because already powered off
                    if vm_local_cdrom[i] in vm_local_resources:
                        vm_local_resources.remove(vm_local_cdrom[i])

            if not DRY_RUN:
                if not check_shutdown(proxmox, node_pve, vm_local_cdrom, "vm","object"):
                    msg = f"Encountered problem with the shutdown of VM with local disks"
                    print(f"\t{msg}")
                    log().error(msg)
                    exit(1)

        
        # List VMs with local disk
        this_node_local_disk = []
        if not VM_LOCAL_DISKS_EXPERT:
            for vm in vm_local_disks:
                if vm['node'] == node_pve:
                    this_node_local_disk.append(vm['vmid'])
            this_node_local_disk.sort()

        # Shutdown VM with local disks if present
        if len(this_node_local_disk) > 0:
            if VERBOSE:
                msg = f"Powering off VM with local disks: {this_node_local_disk}"
            else:
                msg = f"Powering off VM with local disks"

            print(f"\t{msg}")
            log().debug(msg)
            for i in range(len(vm_local_disks)):
                if vm_local_disks[i]['node'] == node_pve:
                    msg = f"Powering off VM {vm_local_disks[i]['vmid']} - {vm_local_disks[i]['name']}"
                    if VERBOSE:
                        print(f"\t{msg}")
                        log().debug(msg)

                    shutdown_single(proxmox, node_pve,vm_local_disks[i]['vmid'], "vm")

                    # Remove the VM from the vm_local_resources if present because already powered off
                    if vm_local_disks[i] in vm_local_resources:
                        vm_local_resources.remove(vm_local_disks[i])
            if not DRY_RUN:
                if not check_shutdown(proxmox, node_pve, vm_local_disks, "vm","object"):
                    msg = f"Encountered problem with the shutdown of VM with local disks"
                    print(f"\t{msg}")
                    log().error(msg)
                    exit(1)

        # list VM with local resources
        this_node_local_resources = []
        for vm in vm_local_resources:
            if vm['node'] == node_pve:
                this_node_local_resources.append(vm['vmid'])
        this_node_local_resources.sort()

        # Shutdown VM with local resources if present
        if len(this_node_local_resources) > 0:
            if VERBOSE:
                msg = f"Powering off VM with local resources: {this_node_local_resources}"
            else:
                msg = f"Powering off VM with local resources"
            print(f"\t{msg}")
            log().debug(msg)
            
            for i in range(len(vm_local_resources)):
                if vm_local_resources[i]['node'] == node_pve:
                    msg = f"Powering off VM {vm_local_resources[i]['vmid']} - {vm_local_resources[i]['name']}"
                    if VERBOSE:
                        print(f"\t{msg}")
                        log().debug(msg)
                    shutdown_single(proxmox, node_pve,vm_local_resources[i]['vmid'],"vm")
            if not DRY_RUN:
                if not check_shutdown(proxmox, node_pve, vm_local_resources, "vm", "object"):
                    msg = f"Encountered problem with the shutdown of VM with local resources"
                    print(f"\t{msg}")
                    log().error(msg)
                    exit(1)
        
        # Create placement list
        survivor_hosts = my_pve_hosts[:]
        survivor_hosts.remove(node_pve)
        vm_placement_list = []

        # Set maintenance mode (if cluster)
        if is_cluster:
            msg = f"Putting {node_pve} in maintenance mode"
            print(f"\t{msg}")
            log().info(msg)

            if VERBOSE:
                msg = f"HA policies will evacuate VMs/CTs: {thisnode_ha_vms}"
            else:
                msg = "HA policies will evacuate VMs/CTS"
            print(f"\t{msg}")
            log().info(msg)
            
            if not DRY_RUN:
                change_maintenace(node_pve, node_ip,"enable")
        
                #new_prox_session = prox_session(survivor_hosts[0])
                # Check when HA-migrate is finished
                check_migration_ha(proxmox, node_ip, node_pve, thisnode_ha_vms,"off")
           
                # Sleep 30 seconds to allow HA-operations to be completed
                msg = "Sleeping 30 secs to complete HA operations"
                print(f"\t{msg}")
                log().info(msg)
                time.sleep(30)
       
        # Refresh list of running VMs
        vm_running_list = get_running(proxmox, node_pve, "vm")
        vm_running_list.sort()

        if len(vm_running_list) > 0:
            msg = "Evacuating VMs"
            print(f"\t{msg}")
            log().info(msg)

            # Removing HA VMs, local resources and local disks VM from the running list because already migrated before (ONLY IN DRY-RUN TO BETTER SIMULATE OUTPUT)
            if DRY_RUN:
                for i in range(len(thisnode_ha_vms)):
                    type,id = thisnode_ha_vms[i].split(":")
                    if type.lower() == "vm":
                        vm_running_list.remove(int(id))
                for i in this_node_local_cdrom:
                    vm_running_list.remove(i)
                for i in range(len(this_node_local_disk)):
                    vm_running_list.remove(this_node_local_disk[i])
                for i in range(len(this_node_local_resources)):
                    vm_running_list.remove(this_node_local_resources[i])

            # If VMs can be migrated, otherwise shutdown
            if len(survivor_hosts) > 0:
                if VERBOSE:
                    msg = f"Will try to migrate VMs to {survivor_hosts} in sequential mode"
                    print(f"\t{msg}")
                    log().debug(msg)

                vm_placement_list = create_placement_list(proxmox, vm_running_list, survivor_hosts, "vm")
                
                # Migrate VMs
                migrate(proxmox,"vm",vm_placement_list,node_pve, "Forward")
            else:
                if VERBOSE:
                    msg = f"No other host is available, shutting down the VMs: {vm_running_list}"
                    print(f"\t{msg}")
                    log().warning(msg)
                
                # Shutting down all the VMs
                shutdown(proxmox, node_pve, vm_running_list, "vm")

        if not DRY_RUN:
            if not check_shutdown(proxmox, node_pve, vm_running_list, "vm", "vmid"):
                msg = f"Not all the running VMs are powered off on node {node_pve}"
                print(f"\t{msg}")
                log().error(msg)
                exit(1)

            
        ## Containers   
        ct_running_list = get_running(proxmox, node_pve, "ct")
        ct_running_list.sort()

        # Removing HA managed CTs from the running list because already migrated before (ONLY IN DRY-RUN TO BETTER SIMULATE OUTPUT)
        if DRY_RUN:
            for i in range(len(thisnode_ha_vms)):
                type,id = thisnode_ha_vms[i].split(":")
                if type.lower() == "ct":
                    ct_running_list.remove(int(id))
                                           
        if VERBOSE:
            if len(ct_running_list) > 1:
                msg = f"These are the running containers: {ct_running_list}"
            elif len(ct_running_list) == 1:
                msg = f"This is the running container: {ct_running_list}"
            else:
                msg = "No running containers found"
                
            print(f"\t{msg}")
            log().debug(msg)  
        
        if len(ct_running_list) > 0:

            # If containers can be migrated, otherwise shutdown
            if len(survivor_hosts) > 0:
                msg = "Evacuating containers"
                print(f"\t{msg}")
                log().info(msg)
                if VERBOSE:
                    msg = f"Will try to migrate container to {survivor_hosts} in sequential mode"
                    print(f"\t{msg}")
                    log().debug(msg)

                ct_placement_list = create_placement_list(proxmox, ct_running_list, survivor_hosts, "ct")
                
                # Migrate containers
                migrate(proxmox,"ct",ct_placement_list,node_pve, "Forward")
            else:
                if VERBOSE:
                    msg = f"No other host is available, shutting down the container: {ct_running_list}"
                    print(f"\t{msg}")
                    log().warning(msg)
                
                # Shutting down all the CTs
                shutdown(proxmox, node_pve, ct_running_list, "ct")

        if not DRY_RUN:
            if not check_shutdown(proxmox, node_pve, ct_running_list, "ct", "vmid"):
                msg = f"Not all the running containers are powered off on node {node_pve}"
                print(f"\t{msg}")
                log().error(msg)
                exit(1)
            
        if not DRY_RUN:
            if VERBOSE:
                msg = f"Updating firmware for: {node_pve} with the following updates:"
                print(f"\t{msg}")
                log().debug(msg)
                for update in list_updates:
                    msg = update
                    print(f"\t - {msg}")
                    log().debug(msg)
            else:
                msg = f"Updating firmware for: {node_pve}"
                print(f"\t{msg}")
                log().info(msg)

            trigger_firmware_update(uuid, list_updates)

            if not node_status(proxmox, node_pve, TIMEOUT_HOST):
                msg = f"Node {node_pve} is not ready after {TIMEOUT_HOST} seconds. Restore manually the node and run again the program"
                print(msg)
                log().error(msg)
                exit(1)

        else:
            if VERBOSE:
                msg = f"Simulating update procedure for node {node_pve} with the following updates:"
                print(f"\t{msg}")
                log().debug(msg)
                for update in list_updates:
                    msg = update
                    print(f"\t - {msg}")
                    log().debug(msg)
            else:
                msg = f"Simulating update procedure for node {node_pve}"
                print(f"\t{msg}")
                log().info(msg)

        
        # Remove maintenance
        if is_cluster:
            msg = f"Disable maintenance mode for {node_pve}"
            print(f"\t{msg}")
            log().info(msg)

            if VERBOSE:
                msg = f"HA managed resources {thisnode_ha_vms} will be handled by assigned HA policies"
                print(f"\t{msg}")
                log().debug(msg)
            
            if not DRY_RUN:
                change_maintenace(node_pve, node_ip,"disable")
        
                # Check when HA-migrate is finished
                check_migration_ha(proxmox, node_ip, node_pve,thisnode_ha_vms,"on")

        # Restore VMs
        if len(vm_running_list) > 0:
            msg = f"Restoring VMs back on node {node_pve}"
            print(f"\t{msg}")
            log().info(msg)

            # If VMs can be migrated back, otherwise power on
            if len(survivor_hosts) > 0:
                # Migrate VMs Back
                migrate(proxmox, "vm", vm_placement_list, node_pve, "Back")
            else:
                # Restart VMs 
                poweron(proxmox, node_pve, vm_running_list, "vm")
        
        if not DRY_RUN:
            if len(vm_running_list) != 0:
                if not check_poweron(node_pve, vm_running_list, "vm"):
                    msg = f"Not all the VM are powered ON on node {node_pve}. Please check"
                    print(f"\t{msg}")
                    log().error(msg)

              
        if len(ct_running_list) > 0:
            msg = "Restoring containers"
            print(f"\t{msg}")
            log().info(msg)
            
            # If containers can be restored back, otherwise power on
            if len(survivor_hosts) > 0:
                if VERBOSE:
                    msg = f"Migrate back container to {node_pve}"
                    print(f"\t{msg}")
                    log().debug(msg)

                # Migrate containers
                migrate(proxmox,"ct",ct_placement_list,node_pve, "Back")
            else:
                if VERBOSE:
                    msg = "Powering on the container"
                    print(f"\t{msg}")
                    log().warning(msg)
                
                # Power on all the CTs
                if not DRY_RUN:
                    poweron(proxmox, node_pve, ct_running_list, "ct")

        if not DRY_RUN:
            if len(ct_running_list) != 0:
                if not check_poweron(node_pve, ct_running_list, "ct"):
                    msg = f"Not all the containers are powered ON on node {node_pve}. Please check"
                    print(f"\t{msg}")
                    log().error(msg)

        # Restore VM with local cdrom
        if len(this_node_local_cdrom) > 0:
            if VERBOSE:
                msg = f"Powering on VMs with local cdrom {this_node_local_cdrom} on node {node_pve}"
            else:
                msg = "Powering on VMs with local cdrom"
            print(f"\t{msg}")
            log().info(msg)

            if len(this_node_local_cdrom) > 0:
                poweron(proxmox, node_pve, this_node_local_cdrom, "vm")
        
        if not DRY_RUN:
            if len(this_node_local_cdrom) != 0:
                if not check_poweron(node_pve, this_node_local_cdrom, "vm"):
                    msg = f"Not all the VM with local cdrom are powered ON on node {node_pve}. Please check"
                    print(f"\t{msg}")
                    log().error(msg)

        # Restore VM with local disks
        if len(this_node_local_disk) > 0:
            if VERBOSE:
                msg = f"Powering on VMs with local disks {this_node_local_disk} on node {node_pve}"
            else:
                msg = "Powering on VMs with local disks"
            print(f"\t{msg}")
            log().info(msg)
            
            if len(this_node_local_disk) > 0:
                poweron(proxmox, node_pve, this_node_local_disk, "vm")

        if not DRY_RUN:
            if len(this_node_local_disk) != 0:
                if not check_poweron(node_pve, this_node_local_disk, "vm"):
                    msg = f"Not all the VM with local disks are powered ON on node {node_pve}. Please check"
                    print(f"\t{msg}")
                    log().error(msg)

        # Restore VM with local resources
        if len(this_node_local_resources) > 0:
            if VERBOSE:
                msg = f"Powering on with local resources {this_node_local_resources} on node {node_pve}"
            else:
                msg = "Powering on with local resources"
            print(f"\t{msg}")
            log().info(msg)
            
            if len(this_node_local_resources) > 0:
                poweron(proxmox, node_pve, this_node_local_resources, "vm")

        if not DRY_RUN:
            if len(this_node_local_resources) != 0:
                if not check_poweron(node_pve, this_node_local_resources, "vm"):
                    msg = f"Not all the VM with local resources are powered ON on node {node_pve}. Please check"
                    print(f"\t{msg}")
                    log().error(msg)        

    msg =  "Ending program"   
    print(msg)
    log().info(msg)
